// (Watchdog safe version)

//=============================================================================
//  PLAY BTC LIB SOUND       (this is the tristate RA4 version)
//=============================================================================
void play_btc_lib_sound(unsigned char thesound)
{
	//----------------------------------------------------
	// var thesound tells which sound to play (0-255)
	//
	// The external I2C eeprom holds; a 1024 byte header
	// which contains 256 pointers to the 256 possible sounds
	// that we can play, followed by the bulk sound data storage.
	// Each pointer is stored in this 4 byte format; X X MSB LSB
	// Each pointer refers to the END point of each sound,
	// and the pointers refer to 32bit boundaries (4 bytes)
	// which allows the 2 byte pointer to address 2 megabit.
	// If a sound does not exist its end pointer is X X 0 0
	//
	// Before we play a sound we get its start and end points from
	// the eeprom, and calculate its length. Then we set the
	// eeprom address to the start of the sound and do one
	// continuous I2C read and BTc play as one operation.
	//----------------------------------------------------

	unsigned char sound_4;			// tracks 4bytes per sound length chunk
	unsigned int start_address;		// where the sound starts in eeprom
	unsigned int sound_length;		// number of 4byte chunks to play

	// first get the START and END pointers for this sound.
	// the 2byte END pointer for a sound is stored at (thesound*4)+2

	// clear the address, default if thesound=0;
	ri2c_add_chip = 0;
	ri2c_add_hi = 0;
	ri2c_add_lo = 2;

	if(thesound)    	// if thesound > 0
	{
		// end pointer for previous sound
		thesound--;

		// load the pointer address
		start_address = ((thesound * 4) + 2);		// 16bit address
		ri2c_add_hi = (start_address / 256);		// hi byte
		ri2c_add_lo = (start_address & 0x00FF);		// lo byte

		// change pointer back to current sound again
		thesound++;
	}

	// now read 2 pointers from the header in the external I2C eeprom
	// first send the address to the eeprom

	ri2c_start_bit();

	data = 0b10100000;	// control byte 1010 AAA WRITE
	ri2c_send_byte();
	data = ri2c_add_hi;   // address of header pointer1
	ri2c_send_byte();
	data = (ri2c_add_lo);
	ri2c_send_byte();

	// tell the eeprom we want to read from it now
	ri2c_start_bit();
	data = 0b10100001;	// control byte 1010 AAA READ
	ri2c_send_byte();

	// read the 2 pointers
	ri2c_receive_byte(ACK);			// put pointer1 in start_address
	start_address = (data * 256);
	ri2c_receive_byte(ACK);
	start_address += data;

	ri2c_receive_byte(ACK);			// read but ignore 2 blank bytes
	ri2c_receive_byte(ACK);

	ri2c_receive_byte(ACK);			// put pointer2 in sound_length
	sound_length = (data * 256);
	ri2c_receive_byte(NO_ACK);		// NO ack on last read!!
	sound_length += data;

	// shut eeprom down
	ri2c_stop_bit();

	// the pointers we just read are referencing 4byte chunks.
	// we must *4 later to get actual eeprom byte addresses

	//----------------------------------------------------
	// exit if a pointer is illegal (ie sound not exist)

	if(!start_address) return;  				// if pointer1 == 0
	if(thesound && !sound_length)	return;		// if pointer2 == 0

	//----------------------------------------------------

	// NOTE!! this is a little tricky because it has to play
	// sound that might start on either eeprom...
	// if the start_address is >= blah, the sound is
	// on the second 512kbit eeprom.


	if(thesound)        // if thesound > 0
	{
	    // get length of the sound, in 4 byte chunks
		sound_length = (sound_length - start_address);

		if(start_address >= (16384-256))	// if start is on 2nd eeprom
		{
            start_address -= 16384;			// subtract 1st eeprom
			ri2c_add_chip = 1;				// select to play 2nd eeprom
		}

		// convert start address to bytes
        start_address = (start_address * 4);
		start_address += 1024;  	// and allow for header
	}
	else      // else thesound==0
	{
		// get the sound length, the first pointer we read
		// was actually sound0 END address
	    sound_length = start_address;

		// start at eeprom start; right after 1024 byte header.
		start_address = 1024;
	}

	// convert start_address from 16bit to two 8bit
	ri2c_add_hi = (start_address / 256);		// hi byte
	ri2c_add_lo = (start_address & 0x00FF);		// lo byte

	//----------------------------------------------------
	// safe! exit if soundlength is zero
	if(!sound_length)	return;

	//----------------------------------------------------
	// now send the start address to the eeprom for the start
	// of sound data, and prepare to i2c read (and play!) the sound

	ri2c_start_bit();

	data = 0b10100000;	// control byte WRITE for eeprom0
	if(ri2c_add_chip) data.F1 = 1;	// if on eeprom1
	ri2c_send_byte();	// control byte; 1010 AAA R/W

	data = ri2c_add_hi;   // address of header pointer1
	ri2c_send_byte();
	data = ri2c_add_lo;
	ri2c_send_byte();

	// tell the eeprom we want to read from it now
	ri2c_start_bit();

	data = 0b10100001;	// control byte READ for eeprom0
	if(ri2c_add_chip) data.F1 = 1;	// if on eeprom1
	ri2c_send_byte();	// control byte; 1010 AAA READ

	//----------------------------------------------------
	// now the eeprom is set to the right address...
	// we can just sequentially read data bytes from it
	// until the sound is finished as these eeproms allow
	// sequential read of entire contents of one eeprom.
	//
	// BUT we don't read data bytes as such, instead
	// this routine uses a fast assembler loop to
	// read I2C BITS and output BTc sound BITS.
	// This process is synchronised to TMR2 which sets the
	// BTc sound bitrate (playback rate) at 44.1kHz.
	//
	// important!! if the sound being played crosses the eeprom
	// boundary we need to stop eeprom0 and restart with eeprom1.
	// we detect the boundary by counting up from the start
	// address, if it reaches 65536 (ie if it rolls to 0)
	// we have reached the end of the first eeprom.
	//
	// This section was adapted from assembler and uses
	// C/assembler mix for speed.
	//----------------------------------------------------
	// trimmed size down to 275 insts.

	TMR2 = 0;		// TMR2 is used to generate 44.1kHz bitrate

	sound_4 = 4;	// will play sound in 4 byte chunks

	// loop until all bytes are played
	play_lib_ri2c_8_bits:
	asm {

		// all vars we use in the asm code are in bank 0
		// so force to bank0 while in this loop

		bcf	STATUS, RP1		;

		// need to release SDA first! because when we get here
		// SDA is held lo by the last ack...

		bsf	STATUS, RP0		;
		bsf RBTC_TSDA		;	// release TRISA SDA
		bcf	STATUS, RP0		;

		// prepare to play the 8 data bits (msb first)

		movlw 8				;
		movwf ri2c_i		;

		//----------------------------------------------------
		// play a bit from i2c
	play_lib_ri2c_bit:

		bsf RBTC_SCL;		// start SCL pulse

		// ready to read i2c bit, first wait for TMR2 sync
		btfss TMR2,7		;	// wait for TMR2 bit7 ==1
		goto $-1			;

		// read the bit from SDA, and output it to BTc speaker pin
		btfsc RBTC_SDA		;
		bsf RBTC_BTC		; // set BTc pin hi
		btfss RBTC_SDA		;
		bcf RBTC_BTC		; // set BTc pin lo


		bcf RBTC_SCL		; // end SCL pulse

		// now need to add a bresenham value to TMR2 to make sure
		// it is 114 insts to the next time TMR2.F7 is hi
		// this removes any cumulative error, and gives a
		// bit playback of (20MHz xtal) 5Mhz / 114 = 43859 Hz (44.1 kHz).

		// so we must add a value to TMR2 of (256-114) = 142
		movlw 142			;
		addwf TMR2,f		;

		// see if all 8 bits received yet
		decfsz ri2c_i,f		;
		goto play_lib_ri2c_bit	;	// more bits left to receive!

		//----------------------------------------------------
		// gets here after playing 8 bits.
		// now we need to make a nice quick ack bit so we
		// don't mess up the 44.1kHz sync for the next data bit...

		bcf RBTC_SDA;		// we make i2c ack bit here, (hold SDA low)
		bsf	STATUS, RP0;	// make SDA an output too!
        bcf RBTC_TSDA;
		bcf	STATUS, RP0;
		goto $+1;			//  so eeprom will send more bytes

		bsf RBTC_SCL;		// make a SCL clock pulse
		goto $+1;
		goto $+1;
		bcf RBTC_SCL;
   }

	// we played 8 sound bits and made an ack!

	// next! check eeprom crossover, ie test if we finished
	// all of eeprom0 and need to keep playing on eeprom1

	start_address++;

	if(start_address == 0)  // if ==0, we reached end of an eeprom
	{
	    // shut down eeprom currently playing
		// (must do a receive byte with NO ack before it is safe
		// to shut down eeprom, the byte is just wasted)
		
		ri2c_receive_byte(NO_ACK);
		ri2c_stop_bit();

	    // if it was eeprom0 that ended
	    if(!ri2c_add_chip)
	    {
			// setup to start eeprom1 at the start
			ri2c_start_bit();
			data = 0b10100010;	// control byte 1010 AAA WRITE
			ri2c_send_byte();
			data = 0;			// address 0x0000
			ri2c_send_byte();
			ri2c_send_byte();
			// tell the eeprom we want to read from it now
			ri2c_start_bit();
			data = 0b10100011;	// control byte 1010 AAA READ
			ri2c_send_byte();
	    }
	    else    // else finished eeprom1 !
	    {
            goto play_lib_done;
		}
	}


	// subtract 1 more byte from count and see if we are done
	sound_4--;
	if(!sound_4)
	{
		sound_4 = 4;

		sound_length--;
        if(!sound_length) goto play_lib_done;
		asm clrwdt;
	}
	// sound is not finished! so keep playing
	goto play_lib_ri2c_8_bits;

	//----------------------------------------------------
	// gets here when sound is finished!
	play_lib_done:

	// we need to send a stop bit to the eeprom
	ri2c_receive_byte(NO_ACK);		// make eeprom finished
	ri2c_stop_bit();

    // always make sound driver pin low before we exit
	RBTC_PIN_BTC = 0;
}
//-----------------------------------------------------------------------------


