/******************************************************************************
	TalkBot_Slave2.c      v1.0    29th July 2009 - Roman Black
	Open-source-firmware for TalkBotBrain product from BlackRobotics.com
	This firmware is open-source for use by legitimate owners
	who purchased a TalkBotBrain. All rights to the TalkBot hardware
	circuitry design remain property of BlackRobotics.

	This is TalkBot_Slave2 - it is identical to Slave1 but
	all serial commands RX/TX must be 2400 baud, NOT 19200.
	(Sound download from PC is still at 19200 baud.)

	TalkBot1 uses a PIC 16F628 and 2x 24LC512 i2c eeproms (1Megabit).
	It is a slave PCB for robotics. It will simultaneously drive
	8 RC servos and play speech or other sounds via a 1bit output.
	It is controlled by 1 serial pin with either 1byte or 2byte
	commands, all at 2400 baud;

	 0sssssss (1byte) ascii 'A' to 'Z'; plays the first 26 sounds; 0-25
	 11111111 SOUND (2byte) plays any sound (SOUND); 0-255
	 1ssspppp pppppppp (2byte) move servo (s) 0-7, to pulse width (p) in uS

	After receiving any valid command TalkBot sends a 1byte confirmation
	out to serial TX; for sound playing commands it sends the sound
	number (0-255) AFTER the sound has been played.
	For move servo command it sends the servo number (0-7)
	after the servo pulse width has been changed.

	For more TalkBot applications and FREE Sound Library files;
	 www.TalkBotBrain.com

// NOTE! Set TABS to 4
******************************************************************************/
/*
_BODEN_OFF   // these configs are set in MikroC compiler options.
_PWRTE_ON
_WDT_OFF
_LVP_OFF
_MCLRE_OFF
_HS_OSC
*/
// vars in general operation, main() and int() etc;

#define PIN_LED1 PORTB.F3	// hi=ON, shared with BTc sound out
#define PIN_LED2 PORTB.F2	// lo=ON, shared with USART TX out, disabled if USART on

#define PIN_BUT1 PORTB.F1	// lo=pressed, shared with USART RX in
#define PIN_BUT2 PORTA.F5	// lo=pressed, general use button

unsigned char t0_toggle;	// for period timing with TMR0
unsigned char t0_count;		// period timing, debouncing

unsigned char RXbyte1;		// the 2-byte command received by RX serial
unsigned char RXbyte2;		//

unsigned char whichsound;	// which sound to play when button2 pressed
unsigned int temp16;		// used in main() for any 16bit math

//---------------------------------------------------------
// vars used for the Servo8 system in main() and int()

unsigned char sv8_maskA;		// to write to ports
unsigned char sv8_maskB;

unsigned char sv8_hipulse;		// for servo sequencing
unsigned char sv8_whichservo;
unsigned char sv8_pinmask;

unsigned char sv8_pulselength_msb[8];	// holds 8 servo pulse lengths
unsigned char sv8_pulselength_lsb[8];

//---------------------------------------------------------
// used in RI2C.C for my i2c eeprom functions

unsigned char ri2c_i absolute 0x20;	// MUST be in register bank0
unsigned char data;

unsigned char ri2c_add_hi;		// 16bit address within external i2c eeprom
unsigned char ri2c_add_lo;		//
unsigned char ri2c_add_chip;	//

#define	RBTC_PIN_SDA	PORTA.F4	// SDA pin, i2c eeprom
#define	RBTC_TRIS_SDA	TRISA.F4	// SDA pin, i2c eeprom
#define	RBTC_PIN_SCL	PORTB.F0	// SCL pin, i2c eeprom
#define	RBTC_PIN_BTC	PORTB.F3	// sound output pin

#define NO_ACK	0				// for i2c to control ack bit
#define ACK		1

// functions in RI2C.c

void ri2c_start_bit(void);
void ri2c_stop_bit(void);
void ri2c_send_byte(void);
void ri2c_receive_byte(unsigned char);

//---------------------------------------------------------
// vars in RBTC.c for BTc 1bit sound playback

#define	RBTC_SDA	PORTA,4			// the same 4 pin defines, syntax ASM only
#define	RBTC_TSDA	TRISA,4			//
#define	RBTC_SCL	PORTB,0			//
#define	RBTC_BTC	PORTB,3			//

// functions in RBTC.c
void play_btc_lib_sound(unsigned char thesound);

//---------------------------------------------------------
// vars in RSTREAM.c needed for serial data stream to eeprom

unsigned char timeout;			// timeout when stream is finished
unsigned char pos_in;			// pointer to position in input buffer
unsigned char input_full;		// flag to show we have 16 more input bytes

unsigned char in_buffer[32];	// buffer to hold incoming serial data stream
unsigned char out_buffer[32] absolute 0xB0;	// data being sent out to I2C eeprom

// functions in RSTREAM.c
void poll_serial_in(void);
void receive_stream(void);
void send_stream(void);
void rdelay_ms(unsigned char);	// used for mS delays anywhere

//-----------------------------------------------------------------------------


//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void  interrupt(void)
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
{
	//----------------------------------------------------
	// is TMR1 overflow int!
	//
	// each of the 8 servos have a 16bit variable which represents
	// a direct TMR1 value. If TMR1 is loaded with this value,
	// it will be the correct time for that servo's hi pulse (1mS to 2mS).
	// so the value is really (0 - time) so the TMR1 roll int will occur
	// when we need to end the hi pulse.
	// since this is just for the TalkBot slave, we assume that the
	// 8 pins are all dedicated outs.
	//----------------------------------------------------

	// so the first thing is to turn all servo pins LO
	// clr all the servo pins
	sv8_maskA = (PORTA & 0b11100000); 		// always keep PORTA.F4 low
	sv8_maskB = (PORTB & 0b00001111);


	PORTA = sv8_maskA;       // clear both servo ports together
	PORTB = sv8_maskB;

	// check servo toggle if we turn servo hi or lo here
	if(sv8_hipulse)    // if a servo goes hi pulse
	{
	    sv8_hipulse = 0;

		// select the next servo to operate
		sv8_whichservo++;
	    // set the mask for that servo output pin
	    sv8_pinmask = (sv8_pinmask << 1);

		if(sv8_whichservo.F3)  // wrap at 0-7
		{
		    sv8_whichservo = 0;
            sv8_pinmask = 0b00000001;
		}

		// if that servo is activated, set its pulse to TMR1
		if(sv8_pulselength_msb[sv8_whichservo])    // if pulse > 0, servo is ON
		{
		    // test if PORTA or PORTB
			if(sv8_whichservo > 3)    // if >3 is portb
			{
				T1CON = 0;  // TMR1 off to write to it
				TMR1H = sv8_pulselength_msb[sv8_whichservo];
				TMR1L = sv8_pulselength_lsb[sv8_whichservo];

				sv8_maskB = (sv8_maskB | sv8_pinmask);   // prepare the output bit

				T1CON = 0b00000001;		// TMR1 ON, at 1:1 prescale
				PORTB = sv8_maskB;			// set the output bit
		    }
		    else      // else is porta
		    {

				T1CON = 0;  // TMR1 off to write to it
				TMR1H = sv8_pulselength_msb[sv8_whichservo];
				TMR1L = sv8_pulselength_lsb[sv8_whichservo];


				sv8_maskA = (sv8_maskA | sv8_pinmask);   // prepare the output bit

				T1CON = 0b00000001;		// TMR1 ON, at 1:1 prescale
				PORTA = sv8_maskA;		// set the output bit
		    }
		}
		else    // else that servo is off
		{

			// set a default 1.5mS delay and exit
		    TMR1H = (256-29);    // 1.5mS
	    }

	}
	//----------------------------------------------------
	else            // else is a 1mS delay low pulse
	{
		sv8_hipulse.F0 = 1;

		// all we do is make a approx 1mS delay between servo hi pulses
	    // 5MHz/ticks TMR1 needs a value of about 5000, (19.5 x 256)
	    // just set TMR1H for convenience
		// this assumes that the servos dont care too much what
		// the pulse low time is. Which they don't.

	    TMR1H = (256-20);
	}

	//----------------------------------------------------
	// clear the TMR1 overflow int and exit
	PIR1.TMR1IF = 0;


	// Tested time to do the int;
	// worst case (servo on) is 29 + 48 -2 + 13 to exit = 88 insts.
	// if a servo is off it is 29 + 24 -2 + 13 = 64 insts
	// and the int to clear the pulse; 29 + 8 - 2 + 13 = 48 insts
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
#include "RI2C.c"		// my i2c eeprom functions
//=============================================================================
#include "RSTREAM.c"	// functions to do serial port data streaming
//=============================================================================
#include "RBTC.c"		// functions to do BTc 1bit sound playback
//=============================================================================


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
	//---------------------------------------------
	// setup the PIC port registers etc

	CMCON = 0x07;			// set all pins to digital, comparators off

	PORTA =  0b00000000;	// RA4 must be kept low
	TRISA  = 0b00110000;  	// RA0-3 servos, all outs,
							// RA4 is SDA to eeprom, high imped
							// RA5 is but2 in

	PORTB =  0b00000101;	// make RB2 TX out high not-active
	TRISB  = 0b00000010;   	// RB0 is SCL out, RB1 is RX in
							// RB2 is TX out, RB3 is sound out
	                        // RB4-7 servos, all outs

	//-------------------------------------------------
 	// TIMER setups (PIC is 20MHz xtal)

	// TMR0 is used for general timing; mS delays, debouncing
	OPTION_REG = 0b10000111;	// TMR0 ON, at 256:1, PORTB pullups OFF

	// TMR1 is used in interrupt for Servo8 system
	T1CON = 0b00000001;		// TMR1 ON, at 1:1 (5MHz)

	// TMR2 is used in BTc sound playback to control bitrate
	T2CON = 0b00000100;		// TMR2 ON, at 1:1 (5MHz)

   	//-------------------------------------------------
	// now TMR0 is operating, make a small delay to let PIC pins settle
	rdelay_ms(100);

	// and make a LED flash (and distinctive sound beep) to show
	// everything powered up ok. we can re-use vars t0_count
	// t0_toggle and whichsound here as they are not needed yet.

	whichsound = 3;			// 3 beeps
	while(whichsound)
	{
		t0_count = 25;  	// of 25 LED pulses
		t0_toggle = 6;		// starting at 6mS period
		while(t0_count)
		{
			// pulse the LED (and sound! is on same pin)
			rdelay_ms(t0_toggle);   // LED off period
			PIN_LED1 = 1;
			rdelay_ms(1);			// LED on period
			PIN_LED1 = 0;

			if(t0_toggle >1) t0_toggle--;
			t0_count--;
		}
		whichsound--;
	}

   	//-------------------------------------------------
	// NOTE! This is Slave2, serial commands are at 2400 baud!!
	// setup USART for serial RX/TX at 2400 baud.

	// for 20MHz xtal 2400 baud;
	TXSTA = 0b00100000;		// TXEN bit5 =1 (enable transmit)
							// BRGH bit2 =0 (low baudrate)
	SPBRG = 129;			// at 20Mhz gives 2400 baud

	// finally enable the USART
	RCSTA = 0b10010000;		// SPEN bit7 =1 enables serial port
							// CREN bit4 =1 allows continuous serial receive

	// note! when USART is ON, we lose control of the RB2 TX pin
	// so we can't light LED2 anymore. It will still flash
	// when TX transmitting. For some applications the USART can
	// be turned off so we can control LED2 again.

   	//-------------------------------------------------

	// servo 8 needs TMR1 interrupt
	PIE1 = 0b00000001;			// TMR1IE on

	INTCON = 0;					// but always start with TMR1 int OFF
	//INTCON = 0b11000000;		// global ints on, TMR1 (PEIE) on

	// note! TMR1 int is OFF when TalkBot boots up so all
	// servos are off, no pulse (limp).
	// TMR1 int turns ON after we receive the first valid
	// 2byte "move servo" serial RX command.
	
   	//-------------------------------------------------
	// setup any variables
	
	// clear all servo values; _msb == 0, means each servo is limp (no pulse)
	sv8_hipulse = 0;
	sv8_pulselength_msb[0] = 0;		// servo0  (256-30) = 1500uS
    sv8_pulselength_msb[1] = 0;
    sv8_pulselength_msb[2] = 0;
    sv8_pulselength_msb[3] = 0;
    sv8_pulselength_msb[4] = 0;
    sv8_pulselength_msb[5] = 0;
    sv8_pulselength_msb[6] = 0;
    sv8_pulselength_msb[7] = 0;		// servo7

	whichsound = 0;		// sound0 is first to play if button2 pressed

	t0_count = 0;		// clear button2 debounce counter

   	//-------------------------------------------------
	// TalkBot main run loop here
	while(1)
	{

		//-------------------------------------------------
		// if BUT2 is held down for >3 seconds,
		// go into receive serial stream mode
		if(!PIN_BUT2)		// if but2 pressed
		{
			if(TMR0.F7 != t0_toggle.F0)   // if detected a 152Hz toggle
			{
				t0_toggle++;
				if(t0_toggle.F0) t0_count++;

				if(t0_count >= 250)		// if but2 pressed >3 seconds
				{
					t0_count = (152-20-1);	// make it flash LED straight away

					// go into serial receive mode here.
					// first change to 19200 baud!
					RCSTA = 0;
					TXSTA = 0b00100100;		// high BRGH mode
					SPBRG = 64;				// at 20Mhz gives 19231 baud
					RCSTA = 0b10010000;		// SPEN bit7 =1 enables serial port

					// flash LED1 and wait for first serial byte to arrive
					while(1)
					{
						// flash LED1 once per second; 152Hz/152
						if(TMR0.F7 != t0_toggle.F0)
                    	{
							t0_toggle++;
							t0_count++;
							if(t0_count == (152-20)) PIN_LED1 = 1; // LED1 ON
							if(t0_count >= 152)
							{
								t0_count = 0;
								PIN_LED1 = 0;	// LED1 OFF
							}
						}
						// check for first serial byte received
						if(PIR1.RCIF == 1)
						{
							while(sv8_hipulse);	// wait for all servo pins lo
							INTCON = 0;			// then turn TMR1 int OFF
							receive_stream();
							t0_count = 0;
							whichsound = 0;		// ready to play first sound
						    break;
						}
					}

					// receive is done, change back to 2400 baud!
					RCSTA = 0;
					TXSTA = 0b00100000;		// low BRGH mode
					SPBRG = 129;			// at 20Mhz gives 2400 baud
					RCSTA = 0b10010000;		// SPEN bit7 =1 enables serial port
				}
			}
		}
		else    // else BUT2 is released
		{
			// if BUT2 was quick pressed, play a sound!
			if(t0_count > 4)	// 4 gives small BUT2 debounce
			{
				play_btc_lib_sound(whichsound);		// play a sound
				TXREG = whichsound;		// send sound number out to serial
				whichsound++;		// select next sound (full 0-255 range)
			}
			t0_count = 0;	// reset BUT2 debounce count
		}


	   	//-------------------------------------------------
		// now check if we received a serial command
		if(PIR1.RCIF == 1)  // if serial byte received
		{
			// clear any serial errors
			if(RCSTA.OERR || RCSTA.FERR)
			{
			    RCSTA = 0b00000000;		// reset USART
				RCSTA = 0b10010000;		// USART back on
				RXbyte1 = RCREG;		// clear USART data
				RXbyte1 = RCREG;		// clear USART data
				whichsound = 0;			// set to sound0 again
			}
			else	// else it's a good serial byte
			{

				// see if it is the first or second byte of 2byte command
				if(RXbyte2 == 0)	// if the first byte
				{
					RXbyte1 = RCREG;    // get byte1

					// if it is a 1byte command = 0xxx xxxx
					if(!RXbyte1.F7)
					{
						// 1byte turn off servo int command
						if(RXbyte1==0x01 && INTCON.F7)
						{
							while(sv8_hipulse);	// wait for all servo pins lo
						    INTCON = 0;			// then turn off servo int!
						}

						// process the 1byte command...
						// valid 1byte play sound commands are;
						//  'A' to 'Z' and 'a' to 'z'
						RXbyte1 = (RXbyte1 & 0b11011111);	// 'a' becomes 'A' etc
						RXbyte1 -= 65;		// 'A' becomes 0

						// now only allow 'A' to 'Z' (ie 0 to 25)
						if(RXbyte1 < 26)
						{
							INTCON = 0; 		// turn servo int off first!
							play_btc_lib_sound(RXbyte1);	// play the sound!
							TXREG = RXbyte1;	// echo sound number out to serial
						}
					}
					else	// else is byte1 of a 2byte command
					{
						RXbyte2++;	// signal that 1byte has been received
					}
				}
				else		// else is second byte of a 2 byte command
				{
					RXbyte2 = RCREG;    // get byte2

					// now we have both bytes of a 2byte command.
				    if(RXbyte1 == 0xFF)		// if it is a 2byte play sound command
				    {
						// just play the sound told by byte2
						play_btc_lib_sound(RXbyte2);	// play the sound!
						TXREG = RXbyte2;	// echo sound number out to serial
				    }
				    else 	// else is a 2byte move servo command
				    {
						// extract top 4 of the 12 bits for pulse length
						temp16 = ((RXbyte1 & 0b00001111) * 256);

		                // extract the 3 bits to make RXbyte1 == servo number
					    RXbyte1 = ((RXbyte1 & 0b01110000) / 16);

						// get bottom 8 bits, now have all 12
						temp16 += RXbyte2;

						// temp16 is now the desired pulse length in uS
						// but TMR1 uses 0.2uS ticks, so we need to * 5
						// using this way is smaller code than (temp16 * 5)
						temp16 = (temp16 + (temp16 * 4));

					    // now need to save that number to the pulse length
					    // variables for that servo. These variables must be TMR1
					    // values to count up, so need to do (0 - value)
						temp16 = (0 - temp16);

						// compensate for int latency of 27 instructions
						// that would make the period too long
						temp16 += 27;   // reduce period by 27

						// and extract to 2 bytes to use in TMR1
						sv8_pulselength_lsb[RXbyte1] = (temp16 & 0x00FF);	// lsb
						sv8_pulselength_msb[RXbyte1] = (temp16 / 256);  // msb

						// turn servo interrupt on if it was off!
						if(!INTCON.F7) 	// if int is off
						{
							INTCON = 0b11000000;	// global ints on, TMR1 (PEIE) on
						}
						TXREG = RXbyte1;	// echo servo number out to serial
					}
					RXbyte2 = 0;	// 2byte command has now been completed
				}
			}
		}
	   	//-------------------------------------------------
		// put some NOPs between testing port pins in main loop.
		// this reduces power consumption of the PIC as
		// NOP uses less power than other PIC instructions.

		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;

		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;

		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
		asm NOP;
	   	//-------------------------------------------------
	}
}
//-----------------------------------------------------------------------------






