/******************************************************************************
	TB_TalkTemp.c      22 Sept 2009 - Roman Black
	This is a "talking thermometer" that speaks the temperature
	when the button is pressed, and also speaks the temperature
	when it changes by more than 0.5 degrees.

	Note! This uses a Dallas DS18S20 temperature sensor, connected
	to pin PORTA.F2 (see the schematic)

	Operation; it stays in sleep mode most of the time to save
	power. One temperature conversion/reading is done about every
	4.5 seconds. If the temperature is quite stable, it reduces power
	by only doing temp readings every 14 seconds.
	 Normal mode = 300uA (check temp every 4.5 secs)
	 Low power mode = 160uA (check temp every 14 secs)
	 Speaking = 15mA to 26mA (depends on volume)

	Connections;
	 PORTB.F5 = external button, speak temperature (active low)
	 PORTA.F2 = dallas1-wire input/output (connects to DS18S20)
	  (PORTA.F2 also needs a 10k pullup resistor to +5v)

// NOTE! Set TABS to 4
******************************************************************************/
/*
_BODEN_OFF   // these configs are set in MikroC compiler options.
_PWRTE_ON
_WDT_ON
_LVP_OFF
_MCLRE_OFF
_HS_OSC
*/
// vars in general operation, main() and int() etc;
#define false 0
#define true 1

#define PIN_LED1 PORTB.F3	// hi=ON, shared with BTc sound out
#define PIN_LED2 PORTB.F2	// lo=ON, shared with USART TX out, disabled if USART on

#define PIN_BUT1 PORTB.F1	// lo=pressed, shared with USART RX in
#define PIN_BUT2 PORTA.F5	// lo=pressed, general use button

unsigned char t0_toggle;	// for period timing with TMR0
unsigned char t0_count;		// period timing, debouncing

//unsigned char RXbyte1;		// the 2-byte command received by RX serial
//unsigned char RXbyte2;		// (not used in this version Slave)

unsigned char whichsound;	// which sound to play when button2 pressed

// used for TB_TalkTemp only;
unsigned char low_power;	// flag for using low power mode
unsigned char sequence;		// main loop sequencing
unsigned char temp;			// actual coarse temp in 'C
unsigned char temp_fine;	// actual temp fraction 1/16'C
unsigned char temp_sign;	// temp MSB (is sign for neg temps)

unsigned int btemp;			// is the 16bit temperature reading
unsigned int btemp_last;	// the last 16bit temp reading
unsigned int btemp_ann;		// the last temperature that was announced

//---------------------------------------------------------
// (Servo8 system is not used in this version Slave)

//---------------------------------------------------------
// used in RI2C.C for my i2c eeprom functions

unsigned char ri2c_i absolute 0x20;	// MUST be in register bank0
unsigned char data;

unsigned char ri2c_add_hi;		// 16bit address within external i2c eeprom
unsigned char ri2c_add_lo;		//
unsigned char ri2c_add_chip;	//

#define	RBTC_PIN_SDA	PORTA.F4	// SDA pin, i2c eeprom
#define	RBTC_TRIS_SDA	TRISA.F4	// SDA pin, i2c eeprom
#define	RBTC_PIN_SCL	PORTB.F0	// SCL pin, i2c eeprom
#define	RBTC_PIN_BTC	PORTB.F3	// sound output pin

#define NO_ACK	0				// for i2c to control ack bit
#define ACK		1

// functions in RI2C.c

void ri2c_start_bit(void);
void ri2c_stop_bit(void);
void ri2c_send_byte(void);
void ri2c_receive_byte(unsigned char);

//---------------------------------------------------------
// vars in RBTC.c for BTc 1bit sound playback
unsigned char dc_loop;
unsigned char dc_level;

#define	RBTC_SDA	PORTA,4			// the same 4 pin defines, syntax ASM only
#define	RBTC_TSDA	TRISA,4			//
#define	RBTC_SCL	PORTB,0			//
#define	RBTC_BTC	PORTB,3			//

// functions in RBTC.c
void play_btc_lib_sound(unsigned char thesound);

//---------------------------------------------------------
// vars in RSTREAM.c needed for serial data stream to eeprom

unsigned char timeout;			// timeout when stream is finished
unsigned char pos_in;			// pointer to position in input buffer
unsigned char input_full;		// flag to show we have 16 more input bytes

unsigned char in_buffer[32];	// buffer to hold incoming serial data stream
unsigned char out_buffer[32] absolute 0xB0;	// data being sent out to I2C eeprom

// functions in RSTREAM.c
void poll_serial_in(void);
void receive_stream(void);
void send_stream(void);
void rdelay_ms(unsigned char);	// used for mS delays anywhere

//-----------------------------------------------------------------------------


//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void  interrupt(void)
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
{
	//----------------------------------------------------
	// (interrupt is not used in this version Slave)
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
#include "RI2C.c"		// my i2c eeprom functions
//=============================================================================
#include "RSTREAM.c"	// functions to do serial port data streaming
//=============================================================================
#include "RBTC.c"		// functions to do BTc 1bit sound playback
//=============================================================================

//=============================================================================
//  SPEAK THE TEMP
//=============================================================================
void speak_the_temp(void)
{
	unsigned char z;	// used and trashed for temp speaking
	//-------------------------------------------------
	// speak the temperature reading using the stored sounds
	// note! because of the way language is spoken this
	// function is a bit clumsy...
	//-------------------------------------------------
	/* Sounds listed by number within sound library;
	0   "0"
	1   1
	2   2
	3   3
	4   4
	5   5
	6   6
	7   7
	8   8
	9   9
	10	ten
	11	eleven
	12	twelve
	13	thirteen
	14	fourteen
	15	fifteen
	16	sixteen
	17	seventeen
	18	eighteen
	19	nineteen
	20	twenty
	21	thirty
	22	forty
	23	fifty
	24	sixty
	25	seventy
	26	eighty
	27	ninety
	28	degrees
	29  point
	30	minus
	31	and
	32	hundred	          */
	//-------------------------------------------------
	// first state if temp is negative
	if(temp_sign) play_btc_lib_sound(30);   // "minus"
	//-------------------------------------------------
	// get temp as degrees C, and remove decimal point
	z = (temp / 2);

	// special case for zero
	if(z == 0)
	{
		play_btc_lib_sound(0);		// "0"
		goto stt_end;
	}

	// special case for hundred
	if(z >= 100)
	{
		play_btc_lib_sound(1);		// "1"
		play_btc_lib_sound(32);		// "hundred"
		if(z > 100) play_btc_lib_sound(31);		// "and"
	    z -= 100;
	}

	//-------------------------------------------------
	// special cases ninety to twenty
	if(z >= 90)
	{
		play_btc_lib_sound(27);	// "ninety"
		z -= 90;
		goto stt_done;
	}
	if(z >= 80)
	{
		play_btc_lib_sound(26);	// "eighty"
		z -= 80;
		goto stt_done;
	}
	if(z >= 70)
	{
		play_btc_lib_sound(25);	// "seventy"
		z -= 70;
		goto stt_done;
	}
	if(z >= 60)
	{
		play_btc_lib_sound(24);	// "sixty"
		z -= 60;
		goto stt_done;
	}
	if(z >= 50)
	{
		play_btc_lib_sound(23);	// "fifty"
		z -= 50;
		goto stt_done;
	}
	if(z >= 40)
	{
		play_btc_lib_sound(22);	// "fourty"
		z -= 40;
		goto stt_done;
	}
	if(z >= 30)
	{
		play_btc_lib_sound(21);	// "thirty"
		z -= 30;
		goto stt_done;
	}
	if(z >= 20)
	{
		play_btc_lib_sound(20);	// "twenty"
		z -= 20;
		goto stt_done;
	}
	stt_done:
	//-------------------------------------------------
	// speak the remainder (z must now be less than 20)
	// so speak z if it is 1-19
	if(z) play_btc_lib_sound(z);

	//-------------------------------------------------
	stt_end:
	// speak the .5 if it exists
	if(temp.F0)
	{
		play_btc_lib_sound(29);		// "point"
		play_btc_lib_sound(5);		// "5"
	}

	//-------------------------------------------------
	// always say "degrees" at the end
	play_btc_lib_sound(28);
}
//-----------------------------------------------------------------------------


//=============================================================================
//  START TEMPERATURE
//=============================================================================
void start_temperature(void)
{
	//-------------------------------------------------
	// This sends a CONVERT_T command to the DS18S20
	// which tells it to start getting a temperature reading.
	//-------------------------------------------------
	OW_reset(&PORTA,2);                      // Onewire reset signal
    OW_Write(&PORTA,2,0xCC);                 // Issue command SKIP_ROM
    OW_Write(&PORTA,2,0x44);                 // Issue command CONVERT_T
}
//-----------------------------------------------------------------------------


//=============================================================================
//  READ TEMPERATURE
//=============================================================================
void read_temperature(void)
{
	unsigned char j;
	//-------------------------------------------------
	// Read the last temperature from DS1820 scratchpad.
	// this READ DATA section is adapted from the MicroC examples
	// and uses the MikroC library functions for the one-wire
	// temp sensor; OW_xxxxx
	//-------------------------------------------------

    OW_Reset(&PORTA,2);
    OW_Write(&PORTA,2,0xCC);                 // Issue command SKIP_ROM
    OW_Write(&PORTA,2,0xBE);                 // Issue command READ_SCRATCHPAD
	asm clrwdt;

	// a small delay, like the orig example which uses 400uS
	TMR0 = 0;
	while(TMR0 < 8);  // wait 400uS (each TMR0 tick is about 51uS)
	
    // Next we perform a 9 byte sequential read block;
    // 0    Temperature_LSB  (just the LSB gives 0xFF = +127.5'C)
    // 1    Temperature_MSB (NOTE! MSB only used for -'C values)
    // 2    Temp alarm High (User byte 1)
    // 3    Temp alarm Low (User byte 2)
    // 4    -reserved-
    // 5    -reserved-
    // 6    Count_remain
    // 7    Count_per_C        (dont bother reading the last 2 bytes)
    // 8    CRC

    temp = OW_Read(&PORTA,2);		// Get temperature LSB
    temp_sign = OW_Read(&PORTA,2);	// Get temperature MSB (sign)

    j = OW_Read(&PORTA,2);          // AL hi (wasted)
    j = OW_Read(&PORTA,2);          // AL lo (wasted)
    j = OW_Read(&PORTA,2);          //  (wasted)
    j = OW_Read(&PORTA,2);          //  (wasted)

    temp_fine = OW_Read(&PORTA,2);	// Get Count_remain    fine res bits
	asm clrwdt;

    //------------------------------------------------
	// The temperature results are in 3 bytes;
	//  temp = temperature in 0.5'C steps ie 0xFF = 127.5'C
	//  temp_sign = (0xFF if temp is negative)
	//  temp_fine = fine temp resolution bits resolution 1/16'C
    //------------------------------------------------
	// calc a 16bit temperature in btemp, this fine resolution
	// temperature value is only used to determine change
	// of >= 0.5'C for the automatic speaking system.

	temp_fine = (16 - temp_fine);   	// get the 4 fine bits
	temp_fine = (temp_fine & 0x0F);     // safe, keep just 4 bits

	btemp = (temp / 2);		// get whole degrees
	btemp = (btemp << 4);	// make room for fine bits
	btemp += temp_fine;		// add the 4 fine bits
	btemp -= 0x04;			// subtract 0.25'C (as per datasheet formula)
	
	// variable temp is used for speaking the temperature.
	// if temp is negative, convert it from 2's complement
	// back to a normal binary number, this makes it easier
	// to speak the number.
	if(temp_sign) temp = (0 - temp);	// if negative, convert it

}
//-----------------------------------------------------------------------------



//=============================================================================
//  MAIN
//=============================================================================
void main()
{
	//-------------------------------------------------
	// setup the PIC port registers etc

	CMCON = 0x07;			// set all pins to digital, comparators off

	PORTA =  0b00000000;	// RA4 must be kept low
	TRISA  = 0b00110100;  	// RA0,1,3 are unused outputs
	                        // RA2 is input, from DS18S20
							// RA4 is SDA to eeprom, high imped
							// RA5 is but2 in

	PORTB =  0b00000101;	// make RB2 TX out high not-active
	TRISB  = 0b00100010;   	// RB0 is SCL out, RB1 is RX in
							// RB2 is TX out, RB3 is sound out
							// RB5 input, speak temp external button
							// RB4,6,7 outputs, unused
	//-------------------------------------------------
 	// TIMER setups (PIC is 20MHz xtal)

	// TMR0 is used for general timing; mS delays, debouncing
	OPTION_REG = 0b00000111;	// TMR0 ON, at 256:1, PORTB pullups ON

	// TMR1 is used in interrupt for Servo8 system
	T1CON = 0b00000001;		// TMR1 ON, at 1:1 (5MHz)

	// TMR2 is used in BTc sound playback to control bitrate
	T2CON = 0b00000100;		// TMR2 ON, at 1:1 (5MHz)

   	//-------------------------------------------------
	// now TMR0 is operating, make a small delay to let PIC pins settle
	rdelay_ms(100);

	// and make a LED flash (and distinctive sound beep) to show
	// everything powered up ok. we can re-use vars t0_count
	// t0_toggle and whichsound here as they are not needed yet.

	whichsound = 3;			// 3 beeps
	while(whichsound)
	{
		t0_count = 25;  	// of 25 LED pulses
		t0_toggle = 6;		// starting at 6mS period
		while(t0_count)
		{
			// pulse the LED (and sound! is on same pin)
			rdelay_ms(t0_toggle);   // LED off period
			PIN_LED1 = 1;
			rdelay_ms(1);			// LED on period
			PIN_LED1 = 0;

			if(t0_toggle >1) t0_toggle--;
			t0_count--;
		}
		whichsound--;
	}

   	//-------------------------------------------------
	// (serial RX/TX commands not used in this version Slave)

   	//-------------------------------------------------
	// (TMR1 interrupt is not used in this version Slave)
	INTCON = 0;		// all ints OFF

   	//-------------------------------------------------
	// setup any variables
	
	whichsound = 0;		// sound0 is first to play if button2 pressed
	t0_count = 0;		// clear button2 debounce counter

   	//-------------------------------------------------
	// TB_TalkTemp only;

	rdelay_ms(250);			// small delay to make sure power is stable

	// setup variables for TB_TalkTemp;
	low_power = 0;
	sequence = 0;
	temp = 0;
	temp_sign = 0;

	btemp_ann = 0; 		// this makes it speak when booted up
	btemp_last = 0;

   	//-------------------------------------------------
	// TalkBot main run loop here
	while(1)
	{
		//-------------------------------------------------
		// use the WDT sleep mode to reduce power consumption.
		// sleep lasts about 18mS so 55 sleeps is approx 1 second.
		// we do a temperature conversion (about once per 5 seconds)
		// and read the temperature value 1 second after.

		asm sleep;
		if(low_power)		// if low power mode, do 3 sleeps not 1
		{
			asm sleep;
			asm sleep;
		}

		// request a temperature reading
		if(sequence == 0) start_temperature();
		if(sequence == 55)      // about 1 sec later
		{
		 	read_temperature();

			// if the temp has changed >= 0.5 degree
			// since the last announcement, announce the temp
			// (and speak it twice)
			if( (btemp >= (btemp_ann + 8)) || (btemp <= (btemp_ann - 8)) )
			{
			    speak_the_temp();
				Rdelay_ms(250);
				Rdelay_ms(250);
				speak_the_temp();
			    btemp_ann = btemp;
			}

			// if the temp has changed >= 0.25 degree then
			// keep checking temp every 5 secs, otherwise
			// switch to low power mode (read temp every 15 secs)
			if( (btemp >= (btemp_last + 4)) || (btemp <= (btemp_last - 4)) )
				low_power = false;
			else
			    low_power = true;
			btemp_last = btemp;
		}
		sequence++;

		//-------------------------------------------------
		// if BUT1 is pressed or external button on PORTB.5
		// is pressed then speak the temp.
		if(!PIN_BUT1 || !PORTB.F5)
		{
			speak_the_temp();
		}

		//-------------------------------------------------
		// if BUT2 is held down for >3 seconds,
		// go into receive serial stream mode
		if(!PIN_BUT2)		// if but2 pressed
		{
			t0_count++;
			if(t0_count >= 166)		// if but2 pressed >3 seconds
			{
				t0_count = (152-20-1);	// make it flash LED straight away

				// go into serial receive mode here.
				// first turn USART on at 19200 baud!
				RCSTA = 0;
				TXSTA = 0b00100100;		// high BRGH mode
				SPBRG = 64;				// at 20Mhz gives 19231 baud
				RCSTA = 0b10010000;		// SPEN bit7 =1 enables serial port

				// flash LED1 and wait for first serial byte to arrive
				while(1)
				{
					asm clrwdt;
					// flash LED1 once per second; 152Hz/152
					if(TMR0.F7 != t0_toggle.F0)
                	{
						t0_toggle++;
						t0_count++;
						if(t0_count == (152-20)) PIN_LED1 = 1; // LED1 ON
						if(t0_count >= 152)
						{
							t0_count = 0;
							PIN_LED1 = 0;	// LED1 OFF
						}
					}
					// check for first serial byte received
					if(PIR1.RCIF == 1)
					{
						receive_stream();
						t0_count = 0;
						whichsound = 0;		// ready to play first sound
					    break;
					}
				}

				// receive is done, turn USART off again...
				RCSTA = 0;
			}
		}
		else    // else BUT2 is released
		{
			// if BUT2 was quick pressed, play a sound!
			if(t0_count > 1)	// 2 gives small BUT2 debounce
			{
				play_btc_lib_sound(whichsound);		// play a sound
				whichsound++;		// select next sound
				if(whichsound > 32) whichsound = 0;    // play 0-32 only
			}
			t0_count = 0;	// reset BUT2 debounce count
		}
	   	//-------------------------------------------------
	}
}
//-----------------------------------------------------------------------------






