/******************************************************************************
	TB_TalkDice.c   - Open Source - Roman Black - Sept 2009
  See; www.TalkBotBrain.com/TalkBot_SourceFiles.htm
	This is a "talking dice" that rolls 1-6 for game playing.
	There are 12 sound samples; 6 (spoken) 1-6, and also 6 sound effects.
  The dice "rolls" 1-6 then speaks 1 of 6 sound sequences.

	This was originally based on the original TalkBot "Slave6"
  random sound player.
	Operation;
	1. press button
	2. it plays one of 6 sound SEQUENCES 
	3. LED "eyes" light up while sound is playing
	4. then LED flashes the roll (1-6) for 2 minutes
  5. Then it goes back into low power sleep mode (150uA)

	The main change to the Slave6 software is that the LED flashes
	the roll (1-6) for 2 minutes after the roll was cast. This
	stops people arguing about what the roll was! It also has a
	modified play_random_sound() function to play 6 sound
  sequences (based on the roll of 1-5).
	After 2 minutes it goes back into low power mode, 140uA,
	with 4x AA batteries will last 6 to 12 months.

	Connections;
	PORTB.F5 is the "roll dice" button, low = pressed (also BUT1)
	PORTA.F3 is the LED (glowing eyes!), high = lit

// NOTE! Set TABS to 2
******************************************************************************/
/*
_BODEN_OFF   // these configs are set in MikroC compiler options.
_PWRTE_ON
_WDT_ON
_LVP_OFF
_MCLRE_OFF
_HS_OSC
*/
// vars in general operation, main() and int() etc;

#define PIN_LED1 PORTB.F3	// hi=ON, shared with BTc sound out
#define PIN_LED2 PORTB.F2	// lo=ON, shared with USART TX out, disabled if USART on

#define PIN_BUT1 PORTB.F1	// lo=pressed, shared with USART RX in
#define PIN_BUT2 PORTA.F5	// lo=pressed, general use button

unsigned char t0_toggle;	// for period timing with TMR0
unsigned char t0_count;		// period timing, debouncing

unsigned char whichsound;	// which sound to play when button2 pressed

// used for TB_TalkDice only;
#define NUMSOUNDS 6	        // used for RNG limits
unsigned char rng_bitcount;	// used to generate a random bit
unsigned char randH;		    // top byte of 16bit random number
unsigned char randL;		    // bottom "
unsigned char rand_temp;    // used in RNG

unsigned char the_roll;     // after a dice roll, is always 1-6
unsigned char flash_num;    // number to flash
unsigned char flash_time;   // times the flashes

unsigned int minute;		// to test if one minute since last roll

//---------------------------------------------------------
// (Servo8 system is not used in this version Slave)

//---------------------------------------------------------
// used in RI2C.C for my i2c eeprom functions

unsigned char ri2c_i absolute 0x20;	// MUST be in register bank0
unsigned char data;

unsigned char ri2c_add_hi;		// 16bit address within external i2c eeprom
unsigned char ri2c_add_lo;		//
unsigned char ri2c_add_chip;	//

#define	RBTC_PIN_SDA	PORTA.F4	// SDA pin, i2c eeprom
#define	RBTC_TRIS_SDA	TRISA.F4	// SDA pin, i2c eeprom
#define	RBTC_PIN_SCL	PORTB.F0	// SCL pin, i2c eeprom
#define	RBTC_PIN_BTC	PORTB.F3	// sound output pin

#define NO_ACK	0				// for i2c to control ack bit
#define ACK		1

// functions in RI2C.c

void ri2c_start_bit(void);
void ri2c_stop_bit(void);
void ri2c_send_byte(void);
void ri2c_receive_byte(unsigned char);

//---------------------------------------------------------
// vars in RBTC.c for BTc 1bit sound playback

#define	RBTC_SDA	PORTA,4			// the same 4 pin defines, syntax ASM only
#define	RBTC_TSDA	TRISA,4			//
#define	RBTC_SCL	PORTB,0			//
#define	RBTC_BTC	PORTB,3			//

// functions in RBTC.c
void play_btc_lib_sound(unsigned char thesound);

//---------------------------------------------------------
// vars in RSTREAM.c needed for serial data stream to eeprom

unsigned char timeout;			// timeout when stream is finished
unsigned char pos_in;			// pointer to position in input buffer
unsigned char input_full;		// flag to show we have 16 more input bytes

unsigned char in_buffer[32];	// buffer to hold incoming serial data stream
unsigned char out_buffer[32] absolute 0xB0;	// data being sent out to I2C eeprom

// functions in RSTREAM.c
void poll_serial_in(void);
void receive_stream(void);
void send_stream(void);
void rdelay_ms(unsigned char);	// used for mS delays anywhere

//-----------------------------------------------------------------------------


//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void  interrupt(void)
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
{
	//----------------------------------------------------
	// (interrupt is not used in this version Slave)
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
#include "RI2C.c"		// my i2c eeprom functions
//=============================================================================
#include "RSTREAM.c"	// functions to do serial port data streaming
//=============================================================================
#include "RBTC.c"		// functions to do BTc 1bit sound playback
//=============================================================================

//=============================================================================
//  PLAY_RANDOM_SOUND
//=============================================================================
void play_random_sound(void)
{
	//-------------------------------------------------
	// Used in TalkBot Slave6 only!
	// This uses a simple random number generator to
	// choose one of the sounds, and play it.
	//
	// 16bit random number generator is taken from Microchip's
	// AN544 math routines appnote "psuedo random number generator"
	// Note! We also add a real world 3bit variable from free-running TMR2
	// so that the random sound is different each time the TalkBot
	// is turned on.
	//-------------------------------------------------
	// 16bit psuedo random number generator (Microchip AN544);
	// 1. XOR bits 15^14 into A
	// 2. XOR bits 12^3 into B
	// 3. XOR bits A^B into C
	// 4. left shift rand var 1 bit
	// 5. put C in bit 0
	//-------------------------------------------------

	// generate 8 RNG bits;
	rng_bitcount = 8;
	while(rng_bitcount)
	{
		// 1. XOR bits 15^14 into A
		rand_temp = 0;
		if(randH.F7 != randH.F6) rand_temp.F2 = 1;

		// 2. XOR bits 12^3 into B
		if(randH.F4 != randL.F3) rand_temp.F1 = 1;

		// 2. XOR bits A^B into C
		if(rand_temp.F2 != rand_temp.F1) rand_temp.F0 = 1;

		// 4. left shift rand var 1bit
		asm CLRF STATUS			;
		asm RLF randL,f			;
		asm RLF randH,f			;

		// 5. put C in bit 0
		if(rand_temp.F0) randL++;

		rng_bitcount--;
	}

	//-------------------------------------------------

	// get the 8 new random bits
	rand_temp = randL;

  // xor a further 2bit random factor taken from TMR2 LSbits
  // TMR2 is free running at 1MHz so the 2 LSbits are very random
  // because it is randomised by user buttonpress.
  rand_temp = (rand_temp ^ (TMR2 & 0x03));

	// convert the 8bit random number to one of
	// the available sounds
	while(rand_temp >= NUMSOUNDS)
	{
		rand_temp -= NUMSOUNDS;
	}
	
	// and play the sound!
	whichsound = rand_temp; // is now 1 of 6 sounds; 0-5
  play_btc_lib_sound(whichsound);       // speak the number
  Rdelay_mS(250);
  play_btc_lib_sound(whichsound+  6);   // play a sound effect
  Rdelay_mS(250);
  play_btc_lib_sound(whichsound);       // speak the number
	
}
//-----------------------------------------------------------------------------

//=============================================================================
//  MAIN
//=============================================================================
void main()
{
	//-------------------------------------------------
	// setup the PIC port registers etc

	CMCON = 0x07;			// set all pins to digital, comparators off

	PORTA =  0b00000000;	// RA4 must be kept low
	TRISA  = 0b00110000;  	// RA0-3 are 4 unused outputs
							// RA4 is SDA to eeprom, high imped
							// RA5 is but2 in

	PORTB =  0b00000101;	// make RB2 TX out high not-active
	TRISB  = 0b00100010;   	// RB0 is SCL out, RB1 is RX in
							// RB2 is TX out, RB3 is sound out
	            // RB5 is the external input button
							// RB4,6,7 unused outputs
	//-------------------------------------------------
 	// TIMER setups (PIC is 20MHz xtal)

	// TMR0 is used for general timing; mS delays, debouncing
	OPTION_REG = 0b00000111;	// TMR0 ON, at 256:1, PORTB pullups ON

	// TMR1 is used in interrupt for Servo8 system
	T1CON = 0b00000001;		    // TMR1 ON, at 1:1 (5MHz)

	// TMR2 is used in BTc sound playback to control bitrate
	T2CON = 0b00000100;		    // TMR2 ON, at 1:1 (5MHz)

   	//-------------------------------------------------
	// now TMR0 is operating, make a small delay to let PIC pins settle
	rdelay_ms(100);

	// and make a LED flash (and distinctive sound beep) to show
	// everything powered up ok. we can re-use vars t0_count
	// t0_toggle and whichsound here as they are not needed yet.

	whichsound = 3;			// 3 beeps
	while(whichsound)
	{
		t0_count = 25;  	// of 25 LED pulses
		t0_toggle = 6;		// starting at 6mS period
		while(t0_count)
		{
			// pulse the LED (and sound! is on same pin)
			rdelay_ms(t0_toggle);   // LED off period
			PIN_LED1 = 1;
			rdelay_ms(1);			// LED on period
			PIN_LED1 = 0;

			if(t0_toggle >1) t0_toggle--;
			t0_count--;
		}
		whichsound--;
	}

  //-------------------------------------------------
	// (serial RX/TX commands not used in this version Slave)

  //-------------------------------------------------
	// (TMR1 interrupt is not used in this version Slave)
	INTCON = 0;		// all ints OFF

  //-------------------------------------------------
	// setup any variables
	
	whichsound = 0;		// sound0 is first to play if button2 pressed
	t0_count = 0;		// clear button2 debounce counter

  //-------------------------------------------------
	// TB_TalkDice demo only!
	
	rdelay_ms(250);			// small delay to make sure power is stable

	randH = 0x30;		// put seed in RNG
	randL = 0x45;		// put seed in RNG

	// setup variables for TB_TalkDice
	minute = 0;
	the_roll = 1;
	flash_num = 0;
	flash_time = 0;


  //-------------------------------------------------
	// TalkBot main run loop here
	while(1)
	{

		//-------------------------------------------------
		// if BUT1 is pressed or external button on PORTB.5
		// is pressed then play a random sound.
		if(!PIN_BUT1 || !PORTB.F5)
		{
			PORTA.F3 = 1;		// LED on
			play_random_sound();
			PORTA.F3 = 0;		// LED off
			the_roll = ((whichsound % 6) + 1);  // set roll to 1-6
			flash_num = 0;
			flash_time = 170;
			minute = 0;
		}

		//-------------------------------------------------
		// if BUT2 is held down for >3 seconds,
		// go into receive serial stream mode
		if(!PIN_BUT2)		// if but2 pressed
		{
			t0_count++;
			if(t0_count >= 166)		// if but2 pressed >3 seconds
			{
				t0_count = (152-20-1);	// make it flash LED straight away

				// go into serial receive mode here.
				// first turn USART on at 19200 baud!
				RCSTA = 0;
				TXSTA = 0b00100100;		// high BRGH mode
				SPBRG = 64;				// at 20Mhz gives 19231 baud
				RCSTA = 0b10010000;		// SPEN bit7 =1 enables serial port

				// flash LED1 and wait for first serial byte to arrive
				while(1)
				{
					asm clrwdt;
					// flash LED1 once per second; 152Hz/152
					if(TMR0.F7 != t0_toggle.F0)
                	{
						t0_toggle++;
						t0_count++;
						if(t0_count == (152-20)) PIN_LED1 = 1; // LED1 ON
						if(t0_count >= 152)
						{
							t0_count = 0;
							PIN_LED1 = 0;	// LED1 OFF
						}
					}
					// check for first serial byte received
					if(PIR1.RCIF == 1)
					{
						receive_stream();
						t0_count = 0;
						whichsound = 0;		// ready to play first sound
					    break;
					}
				}

				// receive is done, turn USART off again...
				RCSTA = 0;
			}
		}
		else    // else BUT2 is released
		{
			// if BUT2 was quick pressed, play a sound!
			if(t0_count > 1)	// 2 gives small BUT2 debounce
			{
				play_btc_lib_sound(whichsound);		// play a sound
				whichsound++;		// select next sound 

				// TB_TalkDice only, auto reset sounds after last sound
				// TB_TalkDice has 12 sounds in the sound library
        if(whichsound >= 12) whichsound=0;
			}
			t0_count = 0;	// reset BUT2 debounce count
		}
 	  //-------------------------------------------------
		// To reduce power consumption, put the PIC in sleep
		// mode. The watchdog will wake it up after 18mS.

		asm sleep;

    //-------------------------------------------------
    // for TB_TalkDice only;
		// this flashes the LED on PORTB.F4 to show the dice roll.
		// it flashes for 2 minutes. the 2 minutes is timed by counting
		// the number of sleep timeouts (18mS each) so there
		// is 3333 sleeps to a minute, so 6666 is 2 minutes.

		// if < 2 minute then do the roll flash
		if(minute < 6666)
		{
			minute++;
			if(flash_time==0) flash_num = the_roll;
			if(flash_time==1 && flash_num>0)
			{
        PORTA.F3 = 1;		// LED on
        flash_num--;
			}
			if(flash_time==3) PORTA.F3 = 0;		// LED off
			if(flash_time==28 && flash_num) flash_time=0;
			flash_time++;
			if(flash_time > 200) flash_time=0;
		}
    else
    {
      PORTA.F3 = 0;		// LED off
    }


    //-------------------------------------------------
	}
}
//-----------------------------------------------------------------------------






