// (Watchdog safe version)

//=============================================================================
//  POLL SERIAL IN
//=============================================================================
void poll_serial_in(void)
{
	//----------------------------------------------------
	// this is a fast function that just checks if a new
	// byte has been received from serial port and if not,
	// checks for timeout (serial stream ended).
	// if serial byte received, we save it in the 32byte
	// input buffer.
	// at 19200 baud, (20MHz PIC) a serial byte comes in every
	// 2600 PIC instructions. So we must call this function
	// more often than that!
	//
	// NOTE! servo TMR1 interrupt must be OFF first!!!!
	//----------------------------------------------------

	// check if we received a new serial byte
	if(PIR1.RCIF)
	{
		timeout = 0;		// reset timeout timer
		in_buffer[pos_in] = RCREG;	// get new data byte
		pos_in++;

		// see if the input buffer is full yet
		if(pos_in == 32)		// if pos_in == 32
		{
			pos_in = 0;
			input_full = 1;	// input_full = true
		}
	}
	else    // if no serial byte, check for timeout
	{
		// if TMR1 has rolled (65536 PIC insts), inc timeout
		if(PIR1.TMR1IF) 		// gets here 76Hz
		{
			PIR1.TMR1IF = 0;
			timeout++;

			// flash LED at 76Hz/8
			t0_toggle++;
			PIN_LED1 = 0;
			if(t0_toggle.F3)
			{
				t0_toggle = 0;
				PIN_LED1 = 1;
			}
		}
		asm clrwdt;
	}
	//----------------------------------------------------
}
//-----------------------------------------------------------------------------


//=============================================================================
//  RECEIVE STREAM
//=============================================================================
void receive_stream(void)
{
	//----------------------------------------------------
	// This just loops and receives the whole data stream
	// from USART RX (serial data from PC) and at the same
	// time it writes the whole stream to external eeprom
	// via bit-banged i2c output.
	//
	// When we enter here the first received byte is
	// already in RXREG and RCIF is set.
	//
	// this uses a 32byte (x2) double buffer system
	// to allow the serial input to still be received while
	// the external eeprom takes 5mS to write each
	// parcel of 32 bytes.
	// Serial input is at 19200 baud, which is 1920 bytes/sec.
	// To fill the 32 byte input buffer takes 16.7mS
	// and it takes (worst case) 5mS deadtime for the eeprom write.
	// so we need to transfer the whole 32 bytes to i2c eeprom
	// in less than 11.7mS (while still polling the input
	// stream) which is easily do-able.
	//
	// once started there is no exit, until the stream ends.
	// this is detected by a timeout of 0.8 second with no
	// data bytes received during that time. If the stream
	// exceeds eeprom max 1Mbit (128 kbyte) it keeps receiving
	// but just stops writing to the eeproms.
	//
	// Timeout detecting;
	// TMR1 16bit is free-running timer at 1:1
	// at 20MHz this overflows at 76Hz
	// so anytime it overflows we inc timeout variable.
	// anytime we get a new serial byte we clr timeout.
	// if timeout >= 64 we have detected a 0.8 second timeout.
	//
	// NOTE! TMR1 interrupt must be OFF before we do this.
	//----------------------------------------------------

	unsigned char i;	// used in loops etc

	// reset counts of serial in
	pos_in = 0;
	input_full = 0;

	// clr the address so we store stream to i2c starting at 0
    ri2c_add_chip = 0;
    ri2c_add_hi = 0;
    ri2c_add_lo = 0;

	// reset timer and its roll flag
	timeout = 0;

	// loop here until stream ends
	while(!timeout.F6)      // loop while timeout is < 64
	{
		//---------------------------------------------------
		// keep checking input
		poll_serial_in();

		//---------------------------------------------------
		// check if we have a full input buffer yet
		// if so, copy in->out, then send all 32 bytes out to
		// write to i2c eeprom.
		// note! if both chips have been filled (if chip == 2) we
		// don't try to save any more data.
		if(input_full && ri2c_add_chip < 2)
		{
			//--------------------------------------
		    // copy in_buffer to out_buffer (32 bytes)

			for(i=0; i<32; i++)
			{
    		    out_buffer[i] = in_buffer[i];
			}
			input_full = 0; 	// now we can start receiving input bytes again

			//--------------------------------------
			// always keep checking serial in while we do stuff

			poll_serial_in();   // 19200 baud, 20MHz = 2600 insts/byte

			//--------------------------------------
			// now send the 32 bytes in output buffer out to I2C eeprom!

			// first send the 3 i2c control bytes
			ri2c_start_bit();

			data = 0b10100000;	// control byte WRITE for eeprom0
			if(ri2c_add_chip) data.F1 = 1;	// if on eeprom1

			ri2c_send_byte();	// control byte; 1010 AAA R/W

			data = ri2c_add_hi;
			ri2c_send_byte();	// address hi
			data = ri2c_add_lo;
			ri2c_send_byte();	// address lo

			//--------------------------------------
			// now i2c is ready we can send 32 data bytes to it

			poll_serial_in();   // keeping checking serial

			// inc the eeprom address by 32 more bytes
			// (this address will be used on the NEXT block sent)
			ri2c_add_lo += 32;
			if(ri2c_add_lo == 0)
			{
				ri2c_add_hi++;
				if(!ri2c_add_hi) // if ==0, then 512k eeprom0 full!
				{
					ri2c_add_chip++; // use next eeprom chip
				}
			}
			//--------------------------------------
		    // now send the 32 data bytes

			for(i=0; i<32; i++)
			{
				data = out_buffer[i];	// send a byte out to i2c
   				ri2c_send_byte();

				// poll serial input after every 4 bytes sent
				if(! (i & 0b00000011)) poll_serial_in();
			}
			//--------------------------------------
		    // finally send the stop bit which starts the eeprom write cycle
			// to write all the 32 bytes at once.
		    // this takes up to 5mS (says 24LC512 datasheet)
		    ri2c_stop_bit();
		}
		//---------------------------------------------------
	}
	//----------------------------------------------------
	// we get here after a timeout has occured.

	if(ri2c_add_chip.F1) return;    // if chip==2, both eeproms full

	// there are probably bytes remaining in the input buffer
	// that need to be written out to eeprom!

	if(pos_in)		// if >0, some bytes remain in buffer
	{
		// fill end of buffer with 0
		for(i=pos_in; i<32; i++) in_buffer[i] = 0;

		// then write the input buffer 32 bytes direct to i2c eeprom
		ri2c_start_bit();

		data = 0b10100000;	// control byte WRITE for eeprom0
		if(ri2c_add_chip) data.F1 = 1;	// if on eeprom1
		ri2c_send_byte();	// control byte; 1010 AAA R/W

		data = ri2c_add_hi;
		ri2c_send_byte();	// address hi
		data = ri2c_add_lo;
		ri2c_send_byte();	// address lo

	    for(i=0; i<32; i++) 	// send remainder of the 32 data bytes
	    {
			data = in_buffer[i];
			ri2c_send_byte();
	    }
	    ri2c_stop_bit();
	}

	RBTC_PIN_BTC = 0;	// turn sound pin OFF before exit
	rdelay_ms(10);		// wait 5mS+ for eeprom write to complete
	//----------------------------------------------------
	// thats it! we have received the entire serial stream
	// and stored the whole thing in the eeproms.
}
//-----------------------------------------------------------------------------



//=============================================================================
//  RDELAY_MS
//=============================================================================
void rdelay_ms(unsigned char mS)
{
	//--------------------------------------------------
	// FOR TalkBot project, must be 20MHz PIC with
	// free running TMR0 at 256:1 prescaler
	// delay is based on TMR0.F1 which toggles every
	// 5Mhz / 256 / 2 = 0.1024 mS
	// This function does not write to TMR0 so it can be
	// used for other timing purposes like the servo
	// interrupt.
	//--------------------------------------------------
	unsigned char loops;

	while(mS)
	{
		asm clrwdt;
		loops = 5;				// 5 loops of 0.2048mS = 1mS
		while(loops)
		{
			while(!TMR0.F1);	// wait for bit0 to be set
			while(TMR0.F1);		// wait for bit0 to be clr
			loops--;
		}
		mS--;
	}
}
//-----------------------------------------------------------------------------

//=============================================================================
//  SEND STREAM
//=============================================================================
void send_stream(void)
{
	//----------------------------------------------------
	// this just loops and sends the entire contents of
	// BOTH I2C eeporms out to the usart TX (serial out to PC).
	// each eeprom is 512kbit (so total is 1Mbit).
	// serial is sent at 19200 baud, so the usart is set
	// up the same as for receive stream (also 19200 baud).
	//
	// this takes 1 min 22 secs!
	// Note! turn TMR1 interrupt off first!
	//----------------------------------------------------
	// trimmed to 69 insts.

	// loop and read/send both eeproms out to serial
	for(ri2c_add_chip=0; ri2c_add_chip<2; ri2c_add_chip++)
	{
		//-------------------------------------------
		// set to start address 0 0
		ri2c_add_hi = 0;
    	ri2c_add_lo = 0;

		// send the start address to the eeprom
		ri2c_start_bit();

		data = 0b10100000;	// control byte; 1010 AAA WRITE
		if(ri2c_add_chip) data.F1 = 1;	// if on eeprom1
		ri2c_send_byte();

		data = ri2c_add_hi;
		ri2c_send_byte();	// address hi
		data = ri2c_add_lo;
		ri2c_send_byte();	// address lo

		ri2c_start_bit();	// another start bit ready to read

		data = 0b10100001;	// control byte; 1010 AAA READ
		if(ri2c_add_chip) data.F1 = 1;	// if on eeprom1
		ri2c_send_byte();

		//-------------------------------------------
		// now read and send entire eeprom contents in one stream
		while(1)
		{
			asm clrwdt;
			
			// get 1 eeprom byte and send it
			ri2c_receive_byte(ACK);		// get byte in data variable
			TXREG = data;				// send that byte out to serial TX
            while(TXSTA.TRMT == 0);		// wait until byte was sent

		    // add a small delay of 1 extra stop bit to increase the
			// reliability of data transfer to slow PC UARTs.
			// TMR2 is 1:1 at 5Mhz (20MHz xtal)
		    TMR2 = 1;
		    while(TMR2++);	// 255 ticks is approx 1 stop bit

		    // count 1 more byte sent and check for eeprom done
		    ri2c_add_lo++;
		    if(!ri2c_add_lo)	// if lo rolled back to 0
		    {
		        ri2c_add_hi++;
		        if(!ri2c_add_hi) break;	// if hi rolled, eeprom was done
		    }
		}

		// close down that eeprom!
		ri2c_receive_byte(NO_ACK); 	// must do this before stop bit
		ri2c_stop_bit();
	}
}
//-----------------------------------------------------------------------------



